#include "BuildingManager.h"
#include "Config.h"
#include "InformationManager.h"
#include "UnitProbeManager.h"
#include "ConvertManager.h"
#include <assert.h>
#include "Juno.h"
#include "DrawManager.h"


/* recover builder previous data */
void PlanBuildData::builderRecover()
{
	if (builder == nullptr)
		return; 

	builder->setRole(builder_role);
	builder->setTargetRegion(builder_target_region);
	builder->setTargetBaseLocation(builder_target_base_location);
	builder->setTargetUnit(builder_target_unit);
	builder->setTargetPosition(builder_target_position);
	builder->setPlanBuildData(nullptr); 
}

/* onFrame call: initialize variables */
void BuildingManager::onFrame()
{
	timer_display_y += 10;

	if (!(BWAPI::Broodwar->getFrameCount() == 0 ||
		BWAPI::Broodwar->getFrameCount() - last_frame > Config::BuildingManager::FrequencyFrame))
	{
		return;
	}

	if (timer.getNoStopElapsedTimeInMicroSec() + Config::Timer::BuildingManagerMaxMicroSec > Config::Timer::TotalMicroSec)
		return;
	Timer tt;
	tt.start(); 

	last_frame = BWAPI::Broodwar->getFrameCount(); 

	// before map analyzed, let depot train Probe
	if (!InformationManager::getInstance().isMapAnalyzed())
	{
		return; 
	}

	InformationManager& IM = InformationManager::getInstance(); 

	// here is what we do after map analyzed
	/* constructing building completed */
	for (std::vector<BWAPI::Unit>::iterator iter = constructings.begin();
		iter != constructings.end(); )
	{
		if ((*iter)->isCompleted())
		{
			if ((*iter)->getType() == BWAPI::UnitTypes::Protoss_Nexus)
			{
				BWTA::BaseLocation* bl = BWTA::getNearestBaseLocation((*iter)->getPosition());
				BWTA::Region* region = bl->getRegion();
				for (auto& u : IM.getRegionSelfUnits(IM.getSelfRegion()))
				{
					u->setTargetBaseLocation(bl);
					u->setTargetRegion(region);
				}
				IM.setSelfBaseLocation(bl);
				IM.setSelfRegion(region);
			}

			buildings.push_back(*iter);
			iter = constructings.erase(iter);
		}
		// enemy is attacking constructing building and hp is low
		else
		{
			if ((*iter)->isUnderAttack())
			{
				int hp = (*iter)->getHitPoints() + (*iter)->getShields();
				double damage = 0;
				for (auto& eu : (*iter)->getUnitsInRadius(Config::BuildingManager::ConstructingEnemyRadius,
					BWAPI::Filter::IsEnemy && BWAPI::Filter::CanAttack))
				{
					if (eu->getTarget() == *iter || eu->getOrderTarget() == *iter)
					{
						BWAPI::WeaponType wt = eu->getType().groundWeapon(); 
						damage += wt.damageAmount() * wt.damageFactor() * 
							double(Config::BuildingManager::FrequencyFrame) / double(wt.damageCooldown());
					}
				}
				if (hp < damage*1.2)
				{
					(*iter)->cancelConstruction();
				}
			}
			iter++;
		}
	}

	/* planned building waiting too long or builder destroy */
	for (std::vector<PlanBuildData*>::iterator iter = plannings.begin();
		iter != plannings.end(); )
	{
		DrawManager::getInstance().drawPlanBuilding((*iter)->location, (*iter)->building_type);

		// finish building, delete 
		bool is_built = false;
		for (auto& b : BWAPI::Broodwar->getUnitsOnTile((*iter)->location))
		{
			if (b->getType() == (*iter)->building_type)
			{
				is_built = true;
				(*iter)->builderRecover();
				delete (*iter);
				iter = plannings.erase(iter);
				break;
			}
		}
		if (is_built)
			continue; 

		// wait too long, delete plan
		if (BWAPI::Broodwar->getFrameCount() - (*iter)->command_frame > Config::BuildingManager::PlanBuildingWaitFrame)
		{
			(*iter)->builderRecover();
			delete (*iter); 
			iter = plannings.erase(iter);
			continue; 
		}

		if (!canBuildHere((*iter)->location, (*iter)->building_type) ||
			((*iter)->building_type.isPowerup() && !BWAPI::Broodwar->hasPower((*iter)->location, (*iter)->building_type)))
		{
			if ((*iter)->building_type.isPowerup())
				(*iter)->location = getNearBuildLocation2((*iter)->location_init, (*iter)->building_type, 4, 1);
			else
				(*iter)->location = getNearBuildLocation2((*iter)->location_init, (*iter)->building_type, 4, 0);
			if (!(*iter)->location.isValid())
			{
				(*iter)->builderRecover();
				delete (*iter);
				iter = plannings.erase(iter);
				continue;
			}
		}

		// builder destroy, find new one 
		if (!(*iter)->builder || !(*iter)->builder->getUnit()->exists())
		{
			UnitProbe u = getBuilder(BWAPI::Position((*iter)->location)); 
			
			if (u)
			{
				(*iter)->builder = u;
				(*iter)->builder_role = u->getRole();
				(*iter)->builder_target_region = u->getTargetRegion();
				(*iter)->builder_target_base_location = u->getTargetBaseLocation();
				(*iter)->builder_target_unit = u->getTargetUnit();
				(*iter)->builder_target_position = u->getTargetPosition();
				(*iter)->command_frame = BWAPI::Broodwar->getFrameCount();

				u->setRole(Role::BUILD);
				u->setPlanBuildData(*iter);
			}
			iter++; 
			continue; 
		}

		iter++; 

	}

	tt.stop();
	DrawManager::getInstance().drawTime(10, timer_display_y, Config::BuildingManager::FrequencyFrame,
		"BuildingManager", tt.getElapsedTimeInMicroSec());

}


/* onUnitCreate call: new self building is constructed */
void BuildingManager::onUnitCreate(BWAPI::Unit b)
{
	if (b->getPlayer() != BWAPI::Broodwar->self() || 
		!b->getType().isBuilding())
		return; 

	// before map analyze
	if (!InformationManager::getInstance().isMapAnalyzed())
	{
		map_unanalyze_buildings.push_back(b); 
		return; 
	}

	InformationManager& IM = InformationManager::getInstance(); 

	// here is what we do after map analyze
	// if already built, mean depot  which we haven't stored 
	if (b->isCompleted())
	{
		buildings.push_back(b);
	}
	else
	{
		for (std::vector<PlanBuildData*>::iterator iter = plannings.begin();
			iter != plannings.end();)
		{
			BWAPI::UnitType ut = b->getType(); 
			BWAPI::TilePosition tp = b->getTilePosition(); 
			if ((*iter)->building_type == b->getType() && (*iter)->location == b->getTilePosition())
			{
				(*iter)->builderRecover();
				delete (*iter);
				iter = plannings.erase(iter);
				break;
			}
			else
				iter++;
		}
		constructings.push_back(b);
	}
}


/* onUnitDestroy call: self building is destroyed */
void BuildingManager::onUnitDestroy(BWAPI::Unit b)
{
	if (b->getPlayer() != BWAPI::Broodwar->self() ||
		!b->getType().isBuilding())
		return;

	std::vector<BWAPI::Unit>::iterator iter = find(buildings.begin(),
		buildings.end(),
		b); 
	if (iter != buildings.end())
		buildings.erase(iter); 
	else
	{
		iter = find(constructings.begin(),
			constructings.end(),
			b);
		if (iter != constructings.end())
			constructings.erase(iter);
		else
			assert(false); 
	}
}

/* delete pointer after game */
void BuildingManager::onEnd()
{
	for (std::vector<PlanBuildData*>::iterator iter = plannings.begin();
		iter != plannings.end();)
	{
		delete (*iter); 
		iter = plannings.erase(iter); 
	}
}


/* add plan building item */
void BuildingManager::addPlanBuilding(PlanBuildData* pbd)
{
	plannings.push_back(pbd); 
}


/* remove plan building item */
void BuildingManager::removePlanBuilding(PlanBuildData* pbd)
{
	std::vector<PlanBuildData*>::iterator iter = std::find(plannings.begin(),
		plannings.end(),
		pbd); 
	if (iter != plannings.end())
	{
		(*iter)->builderRecover();
		delete (*iter);
		iter = plannings.erase(iter);
	}
}


/* assign a unit to build certain type of building at given location */
void BuildingManager::buildAtLocation(const BWAPI::TilePosition& p, const BWAPI::UnitType& type)
{
	if (!p.isValid())
		return; 

//	UnitProbe unit = UnitProbeManager::getInstance().getClosestUnit(BWAPI::Position(p), ROLE_NUM, 32*16); 

	UnitProbe unit = getBuilder(BWAPI::Position(p)); 

	if (unit == nullptr || unit->getRole() == Role::BUILD)
		return; 

	PlanBuildData* pbd = new PlanBuildData; 
	pbd->building_type = type; 
	pbd->location = p; 
	pbd->location_init = p; 
	pbd->builder = unit;
	pbd->builder_role = unit->getRole(); 
	pbd->builder_target_region = unit->getTargetRegion(); 
	pbd->builder_target_base_location = unit->getTargetBaseLocation(); 
	pbd->builder_target_unit = unit->getTargetUnit(); 
	pbd->builder_target_position = unit->getTargetPosition(); 
	pbd->command_frame = BWAPI::Broodwar->getFrameCount(); 

	addPlanBuilding(pbd); 
	/* set BUILD role */
	unit->setRole(Role::BUILD); 
	unit->setPlanBuildData(pbd); 
}


/* return base region build pylon location with the idea of extanding powered area */
BWAPI::TilePosition BuildingManager::getBasePylonLocation()
{
	return getNearBuildLocation2(InformationManager::getInstance().getSelfBaseLocation()->getTilePosition(), 
		BWAPI::UnitTypes::Protoss_Pylon, 
		Config::BuildingManager::BaseBuildRadius, 
		-1); 
}


// get region cannon location is far away from enemy buildings, but close to self base
bool BuildingManager::getCannonLocation(BWTA::Region* r, BWAPI::TilePosition& location)
{
	InformationManager& IM = InformationManager::getInstance();
	// island no
	if (!r->isReachable(IM.getSelfRegion()))
	{
		location = BWAPI::TilePosition(-1, -1);
		return false;
	}

	int cannon_range = BWAPI::UnitTypes::Protoss_Photon_Cannon.sightRange();

	std::list<BWTA::Chokepoint*> cp_path = IM.getRegionRegionPath(r, IM.getSelfRegion()); 

	BWAPI::Position source; 
	if (cp_path.empty())
		source = IM.getSelfBaseLocation()->getPosition();
	else
		source = cp_path.front()->getCenter();

	double closet_dist = 1e9;
	BWAPI::Position target_position(-1, -1);
	for (auto& eb : IM.getRegionEnemyBuildings(r))
	{
		double dist = BWAPI::Position(eb.location).getDistance(source);
		if (dist < closet_dist)
		{
			closet_dist = dist;
			target_position = BWAPI::Position(eb.location); 
		}
	}
	for (auto& eu : IM.getRegionEnemyUnits(r))
	{
		double dist = eu->getDistance(source);
		if (dist < closet_dist)
		{
			closet_dist = dist;
			target_position = eu->getPosition();
		}
	}

	if (!target_position.isValid())
	{
		location = BWAPI::TilePosition(-1, -1); 
		return false; 
	}

	BWAPI::TilePosition location_best(-1, -1);

	BWAPI::Position start(target_position); 
	for (auto& cp : cp_path)
	{
		BWAPI::Position cp_center(cp->getCenter()); 
		BWAPI::Position end(cp_center);

		double length = start.getDistance(end);
		double dir_x = double(end.x - start.x) / length;
		double dir_y = double(end.y - start.y) / length;
		for (int i = 0; i < length; i += 64)
		{
			BWAPI::Position p(start + BWAPI::Position(dir_x*i, dir_y*i));
			BWAPI::TilePosition tp(p);

			int num = BWAPI::Broodwar->getUnitsInRadius(p,
				4 * 32,
				BWAPI::Filter::IsOwned && BWAPI::Filter::IsBuilding && BWAPI::Filter::CanAttack).size(); 
			// already build enough cannon
			if (num >= 8)
			{
				location = BWAPI::TilePosition(-1, -1); 
				return false; 
			}

			location_best = getNearBuildLocation2(tp, BWAPI::UnitTypes::Protoss_Photon_Cannon, 3, 0);

			if (!location_best.isValid())
				continue; 

			bool can_build_here = false; 

			// around build some cannon
			num = BWAPI::Broodwar->getUnitsInRadius(BWAPI::Position(location_best),
				2 * 32,
				BWAPI::Filter::IsOwned && BWAPI::Filter::IsBuilding).size(); 
			if (num > 0 && getBuildingUnits(BWAPI::UnitTypes::Protoss_Photon_Cannon).empty())
			{
				location = location_best; 
				return BWAPI::Broodwar->hasPower(location, BWAPI::UnitTypes::Protoss_Photon_Cannon); 
			}

			// got some protect from cannon around
			num = BWAPI::Broodwar->getUnitsInRadius(BWAPI::Position(location_best),
				2 * 32,
				BWAPI::Filter::IsOwned && BWAPI::Filter::IsBuilding && BWAPI::Filter::CanAttack && BWAPI::Filter::IsCompleted).size();
			if (num > 0)
			{
				location = location_best;
				return BWAPI::Broodwar->hasPower(location, BWAPI::UnitTypes::Protoss_Photon_Cannon);
			}
			// no cannon, must far from enemy
			else
			{
				if (BWAPI::Position(location_best).getDistance(target_position) >= cannon_range + 32 * 2 &&
					BWAPI::Broodwar->isWalkable(BWAPI::WalkPosition(location_best)))
				{
					location = location_best; 
					return BWAPI::Broodwar->hasPower(location, BWAPI::UnitTypes::Protoss_Photon_Cannon);
				}
			}
		}
		
		start = cp_center; 
	}

	location = BWAPI::TilePosition(-1, -1); 
	return false; 
}

bool BuildingManager::getFrontlineLocation(BWTA::Region* r, BWAPI::UnitType unit_type, BWAPI::TilePosition& location)
{
	InformationManager& IM = InformationManager::getInstance();
	BuildingManager& BM = BuildingManager::getInstance(); 
	// island no
	if (!r->isReachable(IM.getSelfRegion()))
	{
		location = BWAPI::TilePosition(-1, -1);
		return false;
	}

	Timer time; 
	time.start(); 

	// already build same building, build around
	std::vector<BWAPI::Unit> building_units = getBuildingUnits(unit_type); 
	std::vector<BWAPI::Unit> constructing_units = getConstructingUnits(unit_type); 
	building_units.insert(building_units.end(), constructing_units.begin(), constructing_units.end());
	for (auto& u : building_units)
	{
		location = getNearBuildLocation2(u->getTilePosition(), unit_type, 8, 1); 
		if (location.isValid())
			return true; 
		location = getNearBuildLocation2(u->getTilePosition(), unit_type, 8, 0);
		if (location.isValid())
			return false; 
	}

	int sight_range = unit_type.sightRange();

	std::list<BWTA::Chokepoint*> cp_path = IM.getRegionRegionPath(r, IM.getSelfRegion());
	BWAPI::Position source;
	if (cp_path.empty())
		source = IM.getSelfBaseLocation()->getPosition();
	else
		source = cp_path.front()->getCenter();

	double closet_dist = 1e9;
	BWAPI::Position target_position(-1, -1);
	for (auto& eb : IM.getRegionEnemyBuildings(r))
	{
		double dist = BWAPI::Position(eb.location).getDistance(source);
		if (dist < closet_dist)
		{
			closet_dist = dist;
			target_position = BWAPI::Position(eb.location);
		}
	}
	for (auto& eu : IM.getRegionEnemyUnits(r))
	{
		double dist = eu->getDistance(source);
		if (dist < closet_dist)
		{
			closet_dist = dist;
			target_position = eu->getPosition();
		}
	}

	double t1 = timer.getNoStopElapsedTimeInMicroSec(); 

	if (!target_position.isValid())
	{
		location = BWAPI::TilePosition(-1, -1);
		return false;
	}

	bool reach_frontline = false;
	BWAPI::TilePosition location_best(-1, -1);

	BWAPI::Position start(target_position);
	for (auto& cp : cp_path)
	{
		BWAPI::Position cp_center(cp->getCenter());
		BWAPI::Position end(cp_center);
		double length = start.getDistance(end);
		double dir_x = double(end.x - start.x) / length;
		double dir_y = double(end.y - start.y) / length;
		for (int i = 0; i < length; i += 32)
		{
			BWAPI::Position p(start + BWAPI::Position(dir_x*i, dir_y*i));
			BWAPI::TilePosition tp(p);
			if (!reach_frontline &&
				!BWAPI::Broodwar->getUnitsOnTile(tp, BWAPI::Filter::IsOwned && BWAPI::Filter::IsBuilding).empty())
			{
				reach_frontline = true; 
				continue; 
			}
			if (reach_frontline)
			{
				BWAPI::TilePosition tmp = getNearBuildLocation2(tp, unit_type, 5, 0); 
				if (tmp.isValid())
				{
					location_best = tmp;
					break;
				}
			}
		}
		if (location_best.isValid())
			break;
		else
			start = cp_center;
	}

	double t2 = timer.getNoStopElapsedTimeInMicroSec();

	if (!location_best.isValid())
	{
		location = location_best;
		return false;
	}

	location = location_best;
	if (BWAPI::Broodwar->hasPower(location, unit_type))
		return true;
	else
		return false; 
}


/* return base region build certain unit type location that has power */
BWAPI::TilePosition BuildingManager::getBaseBuildLocation(const BWAPI::UnitType& type)
{
	InformationManager& IM = InformationManager::getInstance(); 

	return getNearBuildLocation2(IM.getSelfBaseLocation()->getTilePosition(),
		type,
		Config::BuildingManager::BaseBuildRadius,
		1); 

/*	InformationManager& IM = InformationManager::getInstance();

	for (int length = 2; length < 1 + 2 * Config::BuildingManager::BaseBuildRadius; length += 2)
	{
		BWAPI::TilePosition tp(IM.getSelfBaseLocation()->getTilePosition() - BWAPI::TilePosition(length / 2, length / 2));
		for (int i = 0; i < 3; i++)
		{
			int dx, dy;
			switch (i)
			{
			case 0:
				dx = 1; dy = 0;
				break;
			case 1:
				dx = 0; dy = 1;
				break;
			case 2:
				dx = -1; dy = 0;
				break;
			case 3:
				dx = 0; dy = -1;
				break;
			}
			for (int j = 0; j <= length - 1; j++)
			{
				if (BWAPI::Broodwar->hasPower(tp, type) && canBuildHere(tp, type))
					return tp;
				tp.x += dx;
				tp.y += dy;
			}
		}
	}
	return BWAPI::TilePosition(-1, -1); */
}

/* get certain type buildings */
std::vector<BWAPI::Unit> BuildingManager::getBuildingUnits(BWAPI::UnitType type)
{
	if (type == BWAPI::UnitTypes::None)
		return buildings; 

	std::vector<BWAPI::Unit> type_buildings(0); 
	for (auto& u : buildings)
	{
		if (u->getType() == type)
			type_buildings.push_back(u); 
	}

	return type_buildings; 
}

std::vector<BWAPI::Unit> BuildingManager::getConstructingUnits(BWAPI::UnitType type)
{
	if (type == BWAPI::UnitTypes::None)
		return constructings;

	std::vector<BWAPI::Unit> type_constructings(0);
	for (auto& u : constructings)
	{
		if (u->getType() == type)
			type_constructings.push_back(u);
	}
	return type_constructings;
}

std::vector<PlanBuildData*> BuildingManager::getPlanningUnits(BWAPI::UnitType type)
{
	if (type == BWAPI::UnitTypes::None)
		return plannings;

	std::vector<PlanBuildData*> type_plannings(0);
	for (auto& u : plannings)
	{
		if (u->building_type == type)
			type_plannings.push_back(u);
	}
	return type_plannings;
}


/* flag_power = -1, don't have power
              = 0, don't care 
			  = 1, must have power*/
BWAPI::TilePosition BuildingManager::getNearBuildLocation2(const BWAPI::TilePosition& location,
	const BWAPI::UnitType& unit_type,
	int radius, 
	int flag_power)
{
	InformationManager& IM = InformationManager::getInstance(); 

	// when build new building, it need some space from other buildings
	int space = Config::BuildingManager::BuildLocationSpace;
	BWAPI::TilePosition tp_space(space, space);

	bool in_self_region = false; 
	if (InformationManager::getInstance().getRegion(BWAPI::Position(location)) ==
		InformationManager::getInstance().getSelfRegion())
		in_self_region = true; 

	bool need_check = false;
	BWAPI::TilePosition tp(location); 
	switch (flag_power)
	{
	case -1:    // don't have power
		if (!BWAPI::Broodwar->hasPower(tp))
			need_check = true;
		break;
	case 0:     // don't care
		need_check = true;
		break;
	case 1:     // must have power
		if (BWAPI::Broodwar->hasPower(tp, unit_type))
			need_check = true;
		break;
	}
	if (need_check)
	{
		bool is_reachable = BWTA::isConnected(tp, IM.getSelfBaseLocation()->getTilePosition()); 
		bool can_build_here = canBuildHere(tp, unit_type);
		bool is_far_mineral = BWAPI::Broodwar->getUnitsInRadius(BWAPI::Position(tp), 32 * 4, BWAPI::Filter::IsMineralField).empty();
		if (is_reachable && can_build_here && is_far_mineral)
		{
			return tp;
		}
	}

	for (int length = 1; length <= 1 + 2 * radius; length += 2)
	{
		BWAPI::TilePosition tp(location - BWAPI::TilePosition(length / 2, length / 2));
		for (int i = 0; i < 4; i++)
		{
			int dx, dy;
			switch (i)
			{
			case 0:
				dx = 1; dy = 0;
				break;
			case 1:
				dx = 0; dy = 1;
				break;
			case 2:
				dx = -1; dy = 0;
				break;
			case 3:
				dx = 0; dy = -1;
				break;
			}
			for (int j = 0; j < length - 1; j++)
			{
				bool need_check = false; 
				switch (flag_power)
				{
				case -1:    // don't have power
					if (!BWAPI::Broodwar->hasPower(tp))
						need_check = true;
					break;
				case 0:     // don't care
					need_check = true;
					break;
				case 1:     // must have power
					if (BWAPI::Broodwar->hasPower(tp, unit_type))
						need_check = true;
					break;
				}

				if (need_check)
				{
					bool is_reachable = BWTA::isConnected(tp, IM.getSelfBaseLocation()->getTilePosition());
					bool can_build_here = canBuildHere(tp, unit_type);
					bool is_far_mineral = true; 
					if (in_self_region)
						is_far_mineral = BWAPI::Broodwar->getUnitsInRadius(BWAPI::Position(tp), 32 * 4, BWAPI::Filter::IsMineralField).empty();
					if (is_reachable && can_build_here && is_far_mineral)
					{
						return tp;
					}
				}

				tp.x += dx;
				tp.y += dy;
			}
		}
	}
	return BWAPI::TilePosition(-1, -1);
}



void BuildingManager::buildBasePylon()
{
	BWAPI::TilePosition location = getBasePylonLocation();
	if (location.isValid())
		buildAtLocation(location, BWAPI::UnitTypes::Protoss_Pylon);
	else
	{
		location = getBaseBuildLocation(BWAPI::UnitTypes::Protoss_Pylon);
		buildAtLocation(location, BWAPI::UnitTypes::Protoss_Pylon);
	}
}

/* make a position has power,
if some pylon is constructing, return directly
otherwise build a pylon at the same location
return if or not a new pylon is built */
bool BuildingManager::makeLocationPower(const BWAPI::TilePosition& location, BWAPI::UnitType unit_type)
{
	if (!location.isValid())
		return false; 

	if (BWAPI::Broodwar->hasPower(location, unit_type))
		return true; 

	bool is_pylon_constructing = false;

	// pylon power range is little more than 8 tiles
	for (auto& u : BWAPI::Broodwar->getUnitsInRadius(BWAPI::Position(location),
		32 * 7,
		BWAPI::Filter::IsOwned && !BWAPI::Filter::IsCompleted))
	{
		if (u->getType() == BWAPI::UnitTypes::Protoss_Pylon)
		{
			is_pylon_constructing = true;
			break;
		}
	}
	// planning build there
	if (!is_pylon_constructing)
	{
		for (auto& u : plannings)
		{
			if (u->location.getDistance(location) < 7)
			{
				is_pylon_constructing = true; 
				break; 
			}
		}
	}

	if (!is_pylon_constructing && 
		InformationManager::getInstance().getMineral() >= BWAPI::UnitTypes::Protoss_Pylon.mineralPrice())
	{
		buildAtLocation(location, BWAPI::UnitTypes::Protoss_Pylon);
		return true;
	}
	else
		return false; 
}

bool BuildingManager::canBuildHere(const BWAPI::TilePosition& tp, const BWAPI::UnitType& type)
{
/*	int left = tp.x * 32; 
	int top = tp.y * 32; 
	int right = tp.x * 32 + type.tileWidth() * 32 - 1; 
	int bottom = tp.y * 32 + type.tileHeight() * 32 - 1; 
	if (!BWAPI::Broodwar->getUnitsInRectangle(left, top, right, bottom,
		(BWAPI::Filter::IsBuilding && !BWAPI::Filter::IsLifted) || 
		(!BWAPI::Filter::IsOwned && !BWAPI::Filter::IsFlying)).empty())
		return false; */
	for (int x = tp.x; x < tp.x + type.tileWidth(); x++)
	{
		for (int y = tp.y; y < tp.y + type.tileHeight(); y++)
		{
			if (!buildable(x, y, type))
				return false; 
		}
	}
	return true; 
}

bool BuildingManager::buildable(int x, int y, const BWAPI::UnitType & type)
{
	//returns true if this tile is currently buildable, takes into account units on tile
	if (!BWAPI::TilePosition(x,y).isValid() || !BWAPI::Broodwar->isBuildable(x, y)) // &&|| b.type == BWAPI::UnitTypes::Zerg_Hatchery
	{
		return false;
	}

	if (!InformationManager::getInstance().isBuildable(x, y))
		return false; 

	if (!BWAPI::Broodwar->isBuildable(x, y))
		return false; 

	if (BWAPI::Broodwar->hasCreep(x, y))
		return false; 

	return true; 

/*	if (BWAPI::Broodwar->getUnitsOnTile(x, y, BWAPI::Filter::IsBuilding && !BWAPI::Filter::IsLifted).empty())
		return true;
	else
		return false;  */
}


UnitProbe BuildingManager::getBuilder(const BWAPI::Position& p)
{
	InformationManager& IM = InformationManager::getInstance(); 

	BWAPI::Unit closest_u = BWAPI::Broodwar->getClosestUnit(p, BWAPI::Filter::IsOwned && BWAPI::Filter::IsWorker); 
	UnitProbe closest_up = UnitProbeManager::getInstance().getUnit(closest_u); 
	if (closest_up && closest_up->getRole() != Role::BUILD)
		return closest_up;
	else
		return nullptr; 

/*	BWTA::Region* region = IM.getRegion(p); 

	int distance = 99999; 
	UnitProbe builder = nullptr; 

	for (auto& u : IM.getRegionSelfUnits(region))
	{
		int d = u->getUnit()->getDistance(p); 
		if (u->getRole() != Role::BUILD && d < distance)
		{
			distance = d; 
			builder = u; 
		}
	}

	if (builder == nullptr)
	{
		builder = UnitProbeManager::getInstance().getClosestUnit(p, Role::ATTACK); 
	}

	return builder; */
}